/* Objective Modula-2 Compiler (objm2c)
 *
 *  @file objm2_parser.h
 *  ObjM2 parser interface
 *
 *  Syntax analysis for ObjM2 source files
 *
 *  Author: Benjamin Kowarsch
 *
 *  Copyright (C) 2009 Sunrise Telephone Systems KK. All rights reserved.
 *
 *  License:
 *
 *  Permission is hereby granted to review and test this software for the sole
 *  purpose of supporting the effort by the licensor to define and develop the
 *  Objective Modula-2 language. It is not permissible under any circumstances
 *  to  use the software  for the purpose  of creating derivative languages or 
 *  dialects.  This permission is valid until 31 December 2009, 24:00h GMT.
 *
 *  Future licensing:
 *
 *  The licensor undertakes to eventually release this software under a proper
 *  open source license  AFTER  the Objective Modula-2 language definition has
 *  been finalised and a conforming and working reference compiler completed.
 *  
 *  Version history:
 *
 *   2.00   2009-03-30   BK   new file from various spin-offs of v.1.x
 */
 
 
#ifndef OBJM2_PARSER_H
#define OBJM2_PARSER_H


#include "common_types.h"


// --------------------------------------------------------------------------
// Opaque parser handle type
// --------------------------------------------------------------------------
//
// WARNING: Objects of this opaque type should only be accessed through this
// public interface.  DO NOT EVER attempt to bypass the public interface.
//
// The internal data structure of this opaque type is HIDDEN  and  MAY CHANGE
// at any time WITHOUT NOTICE. Accessing the internal data structure directly
// other than through the  functions  in this public interface is  UNSAFE and
// may result in an inconsistent program state or a crash.

typedef opaque_t objm2_parser_t;


// --------------------------------------------------------------------------
// Status codes
// --------------------------------------------------------------------------

typedef /* objm2_parser_status_t */ enum {
    OBJM2_PARSER_STATUS_UNDEFINED = -1,
    
    // operation completed successfully
    OBJM2_PARSER_STATUS_SUCCESS = 1,
    
    // invalid pointer to parser object passed
    OBJM2_PARSER_STATUS_INVALID_REFERENCE,
    
    // sourcefile at specified path not found
    OBJM2_PARSER_STATUS_FILE_NOT_FOUND,
    
    // access to specified sourcefile denied
    OBJM2_PARSER_STATUS_FILE_ACCESS_DENIED,
    
    // open file limit reached, unable to open sourcefile
    OBJM2_PARSER_STATUS_OPEN_FILE_LIMIT_REACHED,
    
    // specified pathname length exceeds maximum length
    OBJM2_PARSER_STATUS_PATH_NAME_TOO_LONG,
    
    // circular reference in specified pathname
    OBJM2_PARSER_STATUS_LOOP_IN_PATHNAME,
    
    // any other error while trying to open sourcefile
    OBJM2_PARSER_STATUS_ERROR_OPENING_FILE,
    
    // unable to allocate memory
    OBJM2_PARSER_STATUS_UNABLE_TO_ALLOCATE
} objm2_parser_status_t;


// ---------------------------------------------------------------------------
// Parser option identifiers
// ---------------------------------------------------------------------------

#define OBJM2_PARSER_OPTION_ALL_OPTIONS_CLEARED  0
#define OBJM2_PARSER_OPTION_PROCESS_C_COMMENTS   1
#define OBJM2_PARSER_OPTION_PROCESS_M2_COMMENTS  2
#define OBJM2_PARSER_OPTION_PROCESS_CPP_COMMENTS 4
#define OBJM2_PARSER_OPTION_SYNTAX_CHECK_ONLY    8
#define OBJM2_PARSER_OPTION_ALL_OPTIONS_SET     15


// ---------------------------------------------------------------------------
// Macro:  OBJM2_PARSER_OPTION_IS_SET(option_to_test, option_flags)
// ---------------------------------------------------------------------------
//
// The  replacement expression  evaluates to  true  if option <option_to_test>
// is set in bitset <option_flags>,  otherwise it evaluates to  false.

#define OBJM2_PARSER_OPTION_IS_SET(_option,_flags) ((_option) & (_flags))


// --------------------------------------------------------------------------
// function:  objm2_new_parser(infile, options, status)
// --------------------------------------------------------------------------
//
// Creates  and  returns  a new  objm2  parser  object  associated  with  the
// specified  source file <infile> and  options <options>.  The status of the
// operation is passed back in <status>.
//
// Returns NULL if the parser object could not be created.

objm2_parser_t objm2_new_parser(const char *infile,
                                  uint16_t options,
                     objm2_parser_status_t *status);


// ---------------------------------------------------------------------------
// function:  objm2_parser_build_parse_tree(parser, ptree, status)
// ---------------------------------------------------------------------------
//
// Processes and builds a parse tree for the sourcefile associated with parser
// <parser>.  A  reference  to the parse tree  is passed back in <ptree>.  The
// status of the operation is passed back in <status>.

void objm2_parser_build_parse_tree(objm2_parser_t parser,
                                    objm2_ptree_t *ptree,
                            objm2_parser_status_t *status);


// ---------------------------------------------------------------------------
// function:  objm2_parser_pathname(parser, status)
// ---------------------------------------------------------------------------
//
// Returns the  pathname  of the sourcefile of parser <parser>.  The status of
// the operation is passed back in <status>.
//
// Returns NULL if <parser> is not a valid parser object.

const char *objm2_parser_pathname(objm2_parser_t parser,
                           objm2_parser_status_t *status);


// ---------------------------------------------------------------------------
// function:  objm2_parser_options(parser, status)
// ---------------------------------------------------------------------------
//
// Returns the option set for parser <parser>.  The status of the operation is
// passed back in <status>.
//
// Returns zero if <parser> is not a valid parser object.

cardinal objm2_parser_options(objm2_parser_t parser,
                       objm2_parser_status_t *status);


// ---------------------------------------------------------------------------
// function:  objm2_reset_parser(parser, status)
// ---------------------------------------------------------------------------
//
// Resets the parser to its initialisation status  and  closes its sourcefile.
// The status of the operation is passed back in <status>.

void objm2_reset_parser(objm2_parser_t parser,
                 objm2_parser_status_t *status);


// ---------------------------------------------------------------------------
// function:  objm2_dispose_parser(parser, status)
// ---------------------------------------------------------------------------
//
// Disposes of parser object <parser> and closes its sourcefile if it is open. 
// The status of the operation is passed back in <status>.

void objm2_dispose_parser(objm2_parser_t parser,
                   objm2_parser_status_t *status);


#endif /* OBJM2_PARSER_H */

// END OF FILE