/* Objective Modula-2 Compiler (objm2c)
 *
 *  objm2_symbols.h
 *  Definition of symbols and related types
 *
 *  Author: Benjamin Kowarsch
 *
 *  Copyright (C) 2009 Sunrise Telephone Systems KK. All rights reserved.
 *
 *  License:
 *
 *  Permission is hereby granted to review and test this software for the sole
 *  purpose of supporting the effort by the licensor to define and develop the
 *  Objective Modula-2 language. It is not permissible under any circumstances
 *  to  use the software  for the purpose  of creating derivative languages or 
 *  dialects.  This permission is valid until 31 December 2009, 24:00h GMT.
 *
 *  Future licensing:
 *
 *  The licensor undertakes to eventually release this software under a proper
 *  open source license  AFTER  the Objective Modula-2 language definition has
 *  been finalised and a conforming and working reference compiler completed.
 *  
 *  Version history:
 *
 *   2.00   2009-03-25   BK   new file
 */


#ifndef OBJM2_SYMBOLS_H
#define OBJM2_SYMBOLS_H


#include "common_types.h"
#include "objm2_tokens.h"


// --------------------------------------------------------------------------
// Symbol attribute flags
// --------------------------------------------------------------------------

typedef /* objm2_symbol_flags_t */ struct {
    bool  is_terminal : 1;
    bool  is_reserved_word : 1;                // only applies to keywords
    bool  is_builtin_ident : 1;                // only applies to identifiers
    bool  is_literal : 1;                      // only applies to literals
    bool  is_operator : 1;                     // only applies to operators
    bool  is_unary_operator : 1;               // only applies to operators
    bool  is_relational_operator : 1;          // only applies to operators
    bool  is_first_order_operator : 1;         // only applies to operators
    bool  is_second_order_operator : 1;        // only applies to operators
    bool  is_malformed_literal : 1;            // only applies to literals
    bool  excess_chars_truncated : 1;          // only applies to identifiers
    bool  has_non_7bit_ascii_chars : 1;        // only applies to strings
    bool  has_one_or_more_underscores : 1;     // only applies to identifiers
    bool  has_one_or_more_dollar_signs : 1;    // only applies to identifiers
    bool  may_collide_with_priv_objc_name : 1; // only applies to identifiers
} objm2_symbol_flags_t;

extern const objm2_symbol_flags_t objm2_symbol_null_flags;


// --------------------------------------------------------------------------
// Predefined flags constant with all flags cleared
// --------------------------------------------------------------------------

#define OBJM2_SYMBOL_FLAGS_CLEARED objm2_symbol_null_flags


// --------------------------------------------------------------------------
// Symbol struct
// --------------------------------------------------------------------------
//
// WARNING: Objects of this struct type should  only  be accessed through the
// getter  and setter  macros  defined by this interface.  DO NOT EVER access
// the fields of this struct type directly.
//
// The  internal data structure  of this struct type  MAY CHANGE  at any time
// WITHOUT NOTICE.  Accessing the internal data structure directly other than
// through the  macros  in this public interface is  UNSAFE and may result in
// an inconsistent program state or a crash.

typedef /* objm2_symbol_s */ struct {

    // symbol id
    uint32_t key;
    octet_t token;
    uchar_t offending_char;
      
    // symbol attributes
    objm2_symbol_flags_t flags;
        
    // line, coloumn and nesting levels 
    struct /* counters */ {
        unsigned line : 17; /* 0 .. 128K */
        unsigned col : 12;  /* 0 .. 4095 */
        int paren_nesting_level : 10;  /* -512 .. 511 */
        int bracket_nesting_level : 10;  /* -512 .. 511 */
        int brace_nesting_level : 10;  /* -512 .. 511 */
        bool line_overflow : 1;  /* for future use */
        bool col_overflow : 1;  /* for future use */
        bool paren_nesting_level_overflow : 1;  /* for future use */
        bool bracket_nesting_level_overflow : 1;  /* for future use */
        bool brace_nesting_level_overflow : 1;  /* for future use */
    } counters;

} objm2_symbol_s;

extern const objm2_symbol_s objm2_symbol_null_symbol;


// --------------------------------------------------------------------------
// Predefined symbol constant with all fields cleared
// --------------------------------------------------------------------------

#define OBJM2_SYMBOL_FIELDS_CLEARED objm2_symbol_null_symbol


// --------------------------------------------------------------------------
// Symbol type
// --------------------------------------------------------------------------

typedef objm2_symbol_s *objm2_symbol_t;


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_init(sym)
// --------------------------------------------------------------------------
//
// Initialises all fields of symbol <sym> to zero values

#define objm2_symbol_init(_sym) _sym = OBJM2_SYMBOL_FIELDS_CLEARED;


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_key(sym)
// --------------------------------------------------------------------------
//
// Returns the key field for symbol <sym>

#define objm2_symbol_key(_sym) _sym.key

// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_key(sym, key)
// --------------------------------------------------------------------------
//
// Sets the key field for symbol <sym> to <key>

#define objm2_symbol_set_key(_sym,_key) _sym.key = _key


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_token(sym)
// --------------------------------------------------------------------------
//
// Returns the token field for symbol <sym>

#define objm2_symbol_token(_sym) _sym.token

// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_token(sym, token)
// --------------------------------------------------------------------------
//
// Sets the token field for symbol <sym> to <token>

#define objm2_symbol_set_token(_sym,_token) _sym.token = _token


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_offending_char(sym)
// --------------------------------------------------------------------------
//
// Returns the offending character field for symbol <sym>.  The value is only
// meaningful if the  symbol's token field is set to TOKEN_ILLEGAL_CHARACTER,
// otherwise the offending character is expected to be ASCII_NUL.

#define objm2_symbol_offending_char(_sym) _sym.offending_char


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_offending_char(sym, char)
// --------------------------------------------------------------------------
//
// Sets the  offending character field  of symbol <sym> to <char>.  The value
// is only expected to be meaningful  if the  symbol's token field  is set to
// TOKEN_ILLEGAL_CHARACTER,  otherwise it should be set to ASCII_NUL.

#define objm2_symbol_set_offending_char(_sym,_ch) _sym.offending_char = _ch


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_flag(sym, flag)
// --------------------------------------------------------------------------
//
// Returns the flag <flag> of symbol <sym>

#define objm2_symbol_flag(_sym,_flag) _sym.flags._flag


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_flag(sym, flag)
// --------------------------------------------------------------------------
//
// Sets the flag <flag> of symbol <sym>

#define objm2_symbol_set_flag(_sym,_flag) _sym.flags._flag = true


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_clear_flag(sym, flag)
// --------------------------------------------------------------------------
//
// Clears the flag <flag> of symbol <sym>

#define objm2_symbol_clear_flag(_sym,_flag) _sym.flags._flag = false


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_all_flags(sym)
// --------------------------------------------------------------------------
//
// Returns all flags of symbol <sym>

#define objm2_symbol_all_flags(_sym) _sym.flags


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_copy_all_flags(sym, flags)
// --------------------------------------------------------------------------
//
// Set all flags of symbol <sym> to <flags>

#define objm2_symbol_set_all_flags(_sym,_flags) _sym.flags = _flags


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_clear_all_flags(sym)
// --------------------------------------------------------------------------
//
// Clears all flags of symbol <sym>

#define objm2_symbol_clear_all_flags(_sym) \
    _sym.flags = OBJM2_SYMBOL_FLAGS_CLEARED


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_line_counter(sym)
// --------------------------------------------------------------------------
//
// Returns the line counter field of symbol <sym>

#define objm2_symbol_line_counter(_sym) _sym.counters.line


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_line_counter(sym, line)
// --------------------------------------------------------------------------
//
// Sets the line counter field of symbol <sym> to <line>

#define objm2_symbol_set_line_counter(_sym,_line) _sym.counters.line = _line


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_col_counter(sym)
// --------------------------------------------------------------------------
//
// Returns the coloumn counter field of symbol <sym>

#define objm2_symbol_col_counter(_sym) _sym.counters.col


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_col_counter(sym, col)
// --------------------------------------------------------------------------
//
// Sets the coloumn counter field of symbol <sym> to <col>

#define objm2_symbol_set_col_counter(_sym,_col) _sym.counters.col = _col


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_paren_nesting_level(sym)
// --------------------------------------------------------------------------
//
// Returns the parenthesis nesting level field of symbol <sym>

#define objm2_symbol_paren_nesting_level(_sym) \
    _sym.counters.paren_nesting_level


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_paren_nesting_level(sym, level)
// --------------------------------------------------------------------------
//
// Sets the parenthesis nesting level field of symbol <sym> to <level>

#define objm2_symbol_set_paren_nesting_level(_sym,_level) \
    _sym.counters.paren_nesting_level = _level


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_bracket_nesting_level(sym)
// --------------------------------------------------------------------------
//
// Returns the bracket nesting level field of symbol <sym>

#define objm2_symbol_bracket_nesting_level(_sym) \
    _sym.counters.bracket_nesting_level


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_bracket_nesting_level(sym, level)
// --------------------------------------------------------------------------
//
// Sets the bracket nesting level field of symbol <sym> to <level>

#define objm2_symbol_set_bracket_nesting_level(_sym,_level) \
    _sym.counters.bracket_nesting_level = _level


// --------------------------------------------------------------------------
// Getter macro:  objm2_symbol_brace_nesting_level(sym)
// --------------------------------------------------------------------------
//
// Returns the brace nesting level field of symbol <sym>

#define objm2_symbol_brace_nesting_level(_sym) \
    _sym.counters.brace_nesting_level


// --------------------------------------------------------------------------
// Setter macro:  objm2_symbol_set_brace_nesting_level(sym, level)
// --------------------------------------------------------------------------
//
// Sets the brace nesting level field of symbol <sym> to <level>

#define objm2_symbol_set_brace_nesting_level(_sym,_level) \
    _sym.counters.brace_nesting_level = _level


#endif /* OBJM2_SYMBOLS_H */

// END OF FILE