/* Objective Modula-2 Compiler (objm2c)
 *
 *  @file objm2_symtab.h
 *  ObjM2 symbol table interface
 *
 *  Symbol table to keep track of symbols read
 *
 *  Author: Benjamin Kowarsch
 *
 *  Copyright (C) 2009 Sunrise Telephone Systems KK. All rights reserved.
 *
 *  License:
 *
 *  Permission is hereby granted to review and test this software for the sole
 *  purpose of supporting the effort by the licensor to define and develop the
 *  Objective Modula-2 language. It is not permissible under any circumstances
 *  to  use the software  for the purpose  of creating derivative languages or 
 *  dialects.  This permission is valid until 31 December 2009, 24:00h GMT.
 *
 *  Future licensing:
 *
 *  The licensor undertakes to eventually release this software under a proper
 *  open source license  AFTER  the Objective Modula-2 language definition has
 *  been finalised and a conforming and working reference compiler completed.
 *  
 *  Version history:
 *
 *   2.00   2009-01-31   BK   new file from various spin-offs of v.1.x
 */

#ifndef OBJM2_SYMTAB_H
#define OBJM2_SYMTAB_H


#include "common_types.h"


// --------------------------------------------------------------------------
// Opaque symbol table handle type
// --------------------------------------------------------------------------
//
// WARNING: Objects of this opaque type should only be accessed through this
// public interface.  DO NOT EVER attempt to bypass the public interface.
//
// The internal data structure of this opaque type is HIDDEN  and  MAY CHANGE
// at any time WITHOUT NOTICE. Accessing the internal data structure directly
// other than through the  functions  in this public interface is  UNSAFE and
// may result in an inconsistent program state or a crash.

typedef opaque_t objm2_symtab_t;


// --------------------------------------------------------------------------
// Status codes
// --------------------------------------------------------------------------

typedef /* objm2_symtab_status_t */ enum {
    OBJM2_SYMTAB_STATUS_UNDEFINED = -1,
    OBJM2_SYMTAB_STATUS_SUCCESS = 1,
} objm2_symtab_status_t;


// --------------------------------------------------------------------------
// function:  objm2_new_symtab()
// --------------------------------------------------------------------------
//
// Creates and returns a new objm2 symbol table object
//
// Returns NULL if the symtab object could not be created.

objm2_symtab_t objm2_new_symtab();


// --------------------------------------------------------------------------
// function:  objm2_symtab_add_entry(symtab, ident, hash)
// --------------------------------------------------------------------------
//
// Adds identifier <ident>  with hash value <hash>  to symbol table <symtab>.
// If zero is passed for <hash>  then the hash value will be calculated.  The
// function will execute slightly faster if it does not have to calculate the
// hash value.
//
// NB:  Lexers which calculate hash values  incrementally  while reading from
// the input stream  can gain a small performance advantage if symbols can be
// stored by supplying a hash value provided the same hash algorithm is used.

objm2_symtab_status_t objm2_symtab_add_entry(objm2_symtab_t symtab,
                                                 const char *ident,
                                                   cardinal hash);


// --------------------------------------------------------------------------
// function:  objm2_symtab_entry_exists(symtab, hash)
// --------------------------------------------------------------------------
//
// Returns  true  if an entry  with hash value <hash>  exists in symbol table
// <symtab>,  otherwise returns  false.  The table will  internally  remember
// the entry and a  subsequent  request to retrieve the same entry by calling
// function objm2_symtab_retrieve_entry will have direct access to the entry.

bool objm2_symtab_entry_exists(objm2_symtab_t symtab, cardinal hash);


// --------------------------------------------------------------------------
// function:  objm2_symtab_retrieve_entry(symtab, hash)
// --------------------------------------------------------------------------
//
// Retrieves a table entry in symbol table <symtab> by hash value <hash>  and
// passes its lexeme in parameter <ident>.  NULL is passed in <ident> if the
// entry does not exist.

objm2_symtab_status_t objm2_symtab_retrieve_entry(objm2_symtab_t symtab,
                                                        cardinal hash,
                                                            char *ident);


// --------------------------------------------------------------------------
// function:  objm2_symtab_remove_entry(symtab, hash)
// --------------------------------------------------------------------------
//
// Removes a table entry from symbol table <symtab> by hash value <hash>.

objm2_symtab_status_t objm2_symtab_remove_entry(objm2_symtab_t symtab,
                                                      cardinal hash);


// ---------------------------------------------------------------------------
// function:  objm2_dispose_symtab(lexer)
// ---------------------------------------------------------------------------
//
// Disposes of symbol table <symtab>, removing all its entries.

objm2_symtab_status_t objm2_dispose_symtab(objm2_symtab_t symtab);


#endif /* OBJM2_SYMTAB_H */

// END OF FILE